//
//  TWTwitchStreamer.m
//  Twitch Streamer
//
//  Created by Auston Stewart on 11/7/13.
//  Copyright (c) 2014 Justin.tv, Inc. All rights reserved.
//

#import "TWTwitchStreamer.h"
#import "TWOALAudioController.h"
#import "TWViewController.h"
#import <AVFoundation/AVFoundation.h>
#import <TwitchKit/TwitchKit.h>
#import "openalsupport.h"
#include "twitchsdk.h"
#include "broadcastlistener.h"

@interface TWTwitchStreamer ()

@property (nonatomic, readwrite) TTV_IngestServer ingestServer;
@property (nonatomic, readwrite) TTV_IngestList ingestList;

- (void)startStream;
- (void)beginFrameSubmission;

@end

namespace
{
	using namespace ttv::broadcast;
	
	class LocalBroadcastListener : public ttv::broadcast::IBroadcastListener
	{
	public:
		
		LocalBroadcastListener(TWTwitchStreamer *twitchStreamer)
		{
			mTwitchStreamer = twitchStreamer;
		}
		
		virtual void LoginRequestCompletedCallback(TTV_ErrorCode result, const TTV_AuthToken& authToken)
		{
			NSLog(@"BROADCASTAPI: Login result: %@",TTV_SUCCEEDED(result) ? @"SUCCESS" : @"ERROR");
		}
		
		virtual void GameNameListRequestCompletedCallback(TTV_ErrorCode result, const TTV_GameInfoList& list)
		{
			NSLog(@"BROADCASTAPI: Game list result: %@",TTV_SUCCEEDED(result) ? @"SUCCESS" : @"ERROR");
			
			for (NSUInteger i = 0; i < list.count; ++i)
			{
				NSLog(@"BROADCAST API: Game: %@",[NSString stringWithUTF8String:list.list[i].name]);
			}
		}

		virtual void RunCommercialRequestCompletedCallback(TTV_ErrorCode result)
		{
			NSLog(@"BROADCASTAPI: Run commercial result: %@",TTV_SUCCEEDED(result) ? @"SUCCESS" : @"ERROR");
		}
		
		virtual void RecordingStatusRequestCompletedCallback(TTV_ErrorCode result, const TTV_ArchivingState& archivingState)
		{
			NSLog(@"BROADCASTAPI: Recording status result: %@",TTV_SUCCEEDED(result) ? @"SUCCESS" : @"ERROR");
		}
		
		virtual void StreamInfoUpdateRequestCompletedCallback(TTV_ErrorCode result, const TTV_StreamInfoForSetting& info)
		{
			NSLog(@"BROADCASTAPI: Stream info update result: %@",TTV_SUCCEEDED(result) ? @"SUCCESS" : @"ERROR");
		}
		
		virtual void IngestListReceivedCallback(const TTV_IngestList& list)
		{
			NSLog(@"BROADCASTAPI: Ingest list received");
			
			for (NSUInteger i = 0; i < list.ingestCount; ++i)
			{
				NSString *serverName = [NSString stringWithUTF8String:list.ingestList[i].serverName];
				NSLog(@"BROADCASTAPI: Ingest Server: %@ Default: %@",serverName,list.ingestList[i].defaultServer ? @"YES" : @"NO");
				
				if ([serverName hasPrefix:@"US West: Los Angeles"]) {
					
					NSLog(@"BROADCASTAPI: Setting ingest server to Los Angeles");
					mTwitchStreamer.ingestServer = list.ingestList[i];
				}
			}
		}
		
		virtual void SendActionMetaDataRequestCompletedCallback(TTV_ErrorCode result)
		{
			NSLog(@"BROADCASTAPI: Send action metadata result: %@",TTV_SUCCEEDED(result) ? @"SUCCESS" : @"ERROR");
		}
		
		virtual void StartSpanMetaDataRequestCompletedCallback(TTV_ErrorCode result)
		{
			NSLog(@"BROADCASTAPI: Start span metadata result: %@",TTV_SUCCEEDED(result) ? @"SUCCESS" : @"ERROR");
		}
		
		virtual void EndSpanMetaDataRequestCompletedCallback(TTV_ErrorCode result)
		{
			NSLog(@"BROADCASTAPI: End span metadata result: %@",TTV_SUCCEEDED(result) ? @"SUCCESS" : @"ERROR");
		}
		
		virtual void FrameSubmissionIssueCallback(TTV_ErrorCode result)
		{
			NSLog(@"BROADCASTAPI: Frame submission issue: %@",TTV_SUCCEEDED(result) ? @"SUCCESS" : @"ERROR");
		}
		
		virtual void BroadcastStartedCallback()
		{
			NSLog(@"BROADCASTAPI: Broadcast started");
			
			[mTwitchStreamer beginFrameSubmission];
		}
		
		virtual void BroadcastStoppedCallback()
		{
			NSLog(@"BROADCASTAPI: Broadcast stopped");
		}
		
		virtual void BroadcastStateChangedCallback(TTV_BroadcastControllerState state)
		{
			NSLog(@"BROADCASTAPI: Broadcast state changed: %d",state);
			
			switch (state) {
				case TTV_BC_STATE_READY_TO_BROADCAST:
					[mTwitchStreamer startStream];
					break;
				default:
					break;
			}
		}
		
		virtual void LoggedOutCallback()
		{
			NSLog(@"BROADCASTAPI: Logged out");
		}
		
	private:
		__weak TWTwitchStreamer *mTwitchStreamer;
	};
}


@implementation TWTwitchStreamer
{
	ALCvoid *_sampleBuffer;
	NSDictionary *_configDict;
	BOOL shouldStartStream;
	TTV_AuthToken _authToken;
	TTV_ChannelInfo _channelInfo;
	TTV_IngestList _ingestList;
	uint8_t **_frameBuffers;
	uint8_t **_freeFrameBuffers;
	dispatch_source_t _audioSubmissionSource;
	dispatch_queue_t _audioSubmissionQueue;
	
	NSTimer *_pollTasksTimer;
	NSTimer *_submitAudioTimer;
	
	ttv::ISDKManager *_SDKManager;
	ttv::broadcast::IBroadcaster *_broadcaster;
	LocalBroadcastListener* _broadcastListener;
}

+ (TWTwitchStreamer *) twitchStreamer
{
	static TWTwitchStreamer *twitchStreamer = nil;
    
	static dispatch_once_t pred;
	dispatch_once(&pred, ^{
		twitchStreamer = [[TWTwitchStreamer alloc] init];
	});
    
	return twitchStreamer;
}

- (void)dealloc
{
	[self stopPollingTasks];
	TTV_DiscardSDKManager();
}

- (id)init
{
	if ((self = [super init])) {
		
		if (![self loadConfiguration])
		{
			NSLog(@"TWSTREAMER: Configuration file missing or invalid");
			return nil;
		}
		
		// instantiate the SDK Manager
		_SDKManager = TTV_GetSDKManager();
		if (_SDKManager == nullptr) return nil;
		
		TTV_ErrorCode ret = _SDKManager->Init(nullptr, [(NSString *)_configDict[@"clientId"] UTF8String], nullptr);
		if (TTV_SUCCEEDED(ret))
		{
			// get the broadcast controller reference
			_broadcaster = _SDKManager->GetBroadcaster(nullptr);
			_broadcastListener = new LocalBroadcastListener(self);
			
			if (_broadcaster == nullptr || _broadcastListener == nullptr) return nil;
			
			// initialize the broadcast controller
			ret = _broadcaster->Init(_broadcastListener);
			if (TTV_SUCCEEDED(ret))
			{
				// TTV_SetTraceLevel(TTV_ML_ERROR);
				// TTV_SetTraceChannelLevel("FrameQueue", TTV_ML_INFO);
				_audioSubmissionSource = NULL;
				_channelInfo.size = sizeof(_channelInfo);
			}
		}
		else
		{
			NSLog(@"TWSTREAMER: Unable to initialize Twitch SDK");
			return nil;
		}
	}
	
	return self;
}

- (BOOL)loadConfiguration
{
	_configDict = [NSDictionary dictionaryWithContentsOfFile:[[NSBundle mainBundle] pathForResource:@"sdkconfig" ofType:@"plist"]];
	if (!_configDict)
	{
		return NO;
	}
	
	// We must have a clientId, plus the required parameters for doing authentication using either the password grant flow
	// or the implicit grant flow; alternatively, an auth token can be provided. An output bitrate must also be provided.
	//
	BOOL hasClientId = ((NSString *)_configDict[@"clientId"]).length;
	BOOL hasPasswordGrantParams = ((NSString *)_configDict[@"clientSecret"]).length &&
								  ((NSString *)_configDict[@"username"]).length &&
								  ((NSString *)_configDict[@"password"]).length;
	
	BOOL hasImplicitGrantParams = ((NSString*)_configDict[@"loginRedirectUri"]).length;
	BOOL hasAuthToken = ((NSString *)_configDict[@"oauthToken"]).length;
	BOOL hasOutputBitrate = [((NSNumber *)_configDict[@"outputBitrate"]) intValue];
	
	return hasClientId && (hasPasswordGrantParams || hasImplicitGrantParams || hasAuthToken) && hasOutputBitrate;
}

- (void)pollTasks
{
	TTV_ErrorCode err = _broadcaster->Update();
	if (!TTV_SUCCEEDED(err)) NSLog(@"TWSTREAMER: Error when polling tasks");
}

- (void)startPollingTasks
{
	if (_pollTasksTimer == nil)
		_pollTasksTimer = [NSTimer scheduledTimerWithTimeInterval:.25 target:self selector:@selector(pollTasks) userInfo:nil repeats:YES];
}

- (void)stopPollingTasks
{
	if (_pollTasksTimer) {
		
		[_pollTasksTimer invalidate];
		_pollTasksTimer = nil;
	}
}

- (void)prepareToStream
{
	shouldStartStream = YES;
	[self authenticate];
	[self startPollingTasks];
}

- (void)authenticate
{
	TTV_ErrorCode ret = TTV_EC_SUCCESS;
	
	TWViewController *viewController = [TWViewController viewController];

	if (((NSString *)_configDict[@"oauthToken"]).length) {
		
		TTV_AuthToken authToken = TTV_AuthToken();
		memcpy(authToken.data,[(NSString *)_configDict[@"oauthToken"] UTF8String],[(NSString *)_configDict[@"oauthToken"] lengthOfBytesUsingEncoding:NSUTF8StringEncoding]);
		ret = _broadcaster->Login(authToken);
		[viewController showRunCommercialButton];
	}
	else if ( ((NSString *)_configDict[@"username"]).length && ((NSString *)_configDict[@"password"]).length && ((NSString *)_configDict[@"clientSecret"]).length ) {
		// If a username, password and clientSecret has been provided we'll try to use the Password Credentials Grant Flow. This will only work if the clientId has been
		// whitelisted. This approach is not normally used on iOS and is only here for testing purposes
		//
		ret = _broadcaster->Login([(NSString *)_configDict[@"username"] UTF8String], [(NSString *)_configDict[@"password"] UTF8String], [(NSString *)_configDict[@"clientSecret"] UTF8String]);
		NSLog(_configDict[@"clientId"]);
		
		[viewController showRunCommercialButton];
	}
	else {
		// Register the notification handler for when the login view is dismissed
		[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(receivedLoginViewDismissedNotification:) name:TKLoginViewWasDismissedNotification object:nil];
		
		// Tell the plugin to show the login webview
		[[TwitchKit sharedPlugin] presentLoginViewForClientID:(NSString*)_configDict[@"clientId"] redirectURI:@"http://www.twitch.tv"];
	}
	
	if (!TTV_SUCCEEDED(ret))
	{
		NSLog(@"TWSTREAMER: Unable to request authentication.");
	}
	else NSLog(@"TWSTREAMER: Requesting auth token");
}
		 
- (void)receivedLoginViewDismissedNotification:(NSNotification *) notification
{
	NSString* authTokenString = [[notification userInfo] objectForKey:TKAuthTokenUserInfoKey];
	
	if ([authTokenString length] > 0)
	{
		TWViewController *viewController = [TWViewController viewController];
		[viewController removeLoginButton];
		[viewController showRunCommercialButton];
		
		TTV_AuthToken authToken = TTV_AuthToken();
		memcpy(authToken.data,[authTokenString UTF8String],[authTokenString lengthOfBytesUsingEncoding:NSUTF8StringEncoding]);
		_broadcaster->Login(authToken);
	}
	else
	{
		NSString* errorString = [[notification userInfo] objectForKey:TKErrorUserInfoKey];
		if ([errorString length] > 0)
		{
			NSLog(@"Authentication failed: %@", errorString);
		}
		else
		{
			// NOP - User must have hit 'cancel' on the login view
		}
	}
	
	[[NSNotificationCenter defaultCenter] removeObserver:self name:TKLoginViewWasDismissedNotification object:nil];
}

- (void)startStream
{
	if (_broadcaster->GetCurrentState() == TTV_BC_STATE_READY_TO_BROADCAST && shouldStartStream)
	{
		// Set the game title
		TTV_ErrorCode err = _broadcaster->UpdateStreamInfo("Twitch Streamer", "Playing with the broadcast controller");
		if (!TTV_SUCCEEDED(err))
		{
			NSLog(@"TWSTREAMER: Error requesting stream info update");
		}
		
		// Request game names
		/*
		 err = _broadcaster->RequestGameNameList("Star");
		 if (TTV_FAILED(err))
		 {
			NSLog(@"TWSTREAMER: Unable to request game names");
		 }
		 */
		
		// Use automatic frame capture
		err = _broadcaster->SetGameVideoSubmissionMethod(TTV_BC_GAME_VIDEO_SUBMISSION_METHOD_AUTOMATIC);
		if (TTV_FAILED(err))
		{
			NSLog(@"TWSTREAMER: Unable to set system video capture method");
			return;
		}
		
		// Assisted configuration
		err = _broadcaster->SetVideoParams([(NSNumber *)_configDict[@"outputBitrate"] intValue],
													  kPreferredFramesPerSecond,
													  .1f,
													  [UIScreen mainScreen].bounds.size.height / [UIScreen mainScreen].bounds.size.width);
		if (TTV_FAILED(err))
		{
			NSLog(@"TWSTREAMER: Failed to set recommended video parameters");
			return;
		}
		
		NSLog(@"TWSTREAMER: Output video size: %dx%d",_broadcaster->GetOutputWidth(),_broadcaster->GetOutputHeight());
		
		// Setup the audio parameters
		_broadcaster->SetCaptureMicrophone([_configDict[@"microphoneCapture"] boolValue]);
		_broadcaster->SetGameAudioSubmissionMethod([_configDict[@"openALCapture"] boolValue] ? TTV_BC_GAME_AUDIO_SUBMISSION_METHOD_MANUAL : TTV_BC_GAME_AUDIO_SUBMISSION_METHOD_NONE);
			
		// Start Streaming
		err = _broadcaster->StartBroadcast();
		if (TTV_SUCCEEDED(err)) {
			
			if ([_configDict[@"openALCapture"] boolValue] && [_configDict[@"microphoneCapture"] boolValue]) {
				
				// Set audio levels to avoid clipping if both microphone and OpenAL audio are mixed
				_broadcaster->SetMicrophoneVolume(.9f);
				_broadcaster->SetManualSubmissionVolume(.8f);
			}
		}
		else NSLog(@"TWSTREAMER: Unable to start streaming.");
	}
}

- (void)beginFrameSubmission
{
	TTV_AuthToken authToken;
	TTV_ChannelInfo channelInfo;
	TTV_ErrorCode err;
	
	err = _broadcaster->GetAuthToken(authToken);
	if (TTV_SUCCEEDED(err))
	{
		NSLog(@"TWSTREAMER: Using auth token: %@",[NSString stringWithUTF8String:authToken.data]);
	}
	else NSLog(@"TWSTREAMER: Error getting auth token");
	
	err = _broadcaster->GetChannelInfo(channelInfo);
	if (TTV_SUCCEEDED(err))
	{
		NSLog(@"TWSTREAMER: Username: %@",[NSString stringWithUTF8String:channelInfo.name]);
		NSLog(@"TWSTREAMER: Display name: %@",[NSString stringWithUTF8String:channelInfo.displayName]);
		NSLog(@"TWSTREAMER: Channel URL: %@",[NSString stringWithUTF8String:channelInfo.channelUrl]);
	}
	else NSLog(@"TWSTREAMER: Error getting channel info");
	
	// Start OpenAL Capture
	if ([_configDict[@"openALCapture"] boolValue])
		[[TWOALAudioController sharedAudioController] startCapture];
}

- (void)stopStream
{
	shouldStartStream = NO;
	
	// Stop OpenAL Capture
	[[TWOALAudioController sharedAudioController] stopCapture];
	
	TTV_ErrorCode ret = _broadcaster->StopBroadcast();
	
	if (TTV_FAILED(ret))
	{
		NSLog(@"TWSTREAMER: Failed asynchronous stop");
	}
}

- (void)restartStream
{
	shouldStartStream = YES;
	[self startStream];
}

- (void)shutdown:(BOOL)synchronous
{
	TTV_ErrorCode ret = _broadcaster->Shutdown(synchronous);
	
	if (TTV_FAILED(ret))
	{
		NSLog(@"TWSTREAMER: Failed shutdown");
	}
}

- (void)pause
{
	TTV_ErrorCode ret = _broadcaster->PauseBroadcast();
	
	if (TTV_FAILED(ret))
	{
		NSLog(@"TWSTREAMER: Failed to pause");
	}
}

- (void)resume
{
	TTV_ErrorCode ret = _broadcaster->ResumeBroadcast();
	
	if (TTV_FAILED(ret))
	{
		NSLog(@"TWSTREAMER: Failed to resume");
	}
}

- (void)logout
{
	TTV_ErrorCode ret = _broadcaster->Logout();
	
	if (TTV_FAILED(ret))
	{
		NSLog(@"TWSTREAMER: Failed to log out");
	}
}

- (void)runCommercial
{
	TTV_ErrorCode ret = _broadcaster->RunCommercial(30);
	
	if (TTV_FAILED(ret))
	{
		NSLog(@"TWSTREAMER: Failed to request commercial");
	}
}

- (BOOL)isStreaming
{
	TTV_BroadcastControllerState state = _broadcaster->GetCurrentState();
	return (state == TTV_BC_STATE_BROADCASTING || state == TTV_BC_STATE_PAUSED);
}

- (BOOL)isPaused
{
	return (_broadcaster->GetCurrentState() == TTV_BC_STATE_PAUSED);
}

- (void)submitAudioSamples
{
	TTV_ErrorCode ret;
	ALint availableFrames = [[TWOALAudioController sharedAudioController] captureFrames];
	if (availableFrames)
	{
		ret = _broadcaster->SubmitAudioSamples((const int16_t*)[TWOALAudioController sharedAudioController].sampleBuffer,(uint)availableFrames * 2);
		if (!TTV_SUCCEEDED(ret)) NSLog(@"TWSTREAMER: Error submitting audio samples: %d",ret);
	}
}

- (void)setIngestServer:(TTV_IngestServer)ingestServer
{
	TTV_ErrorCode ec = _broadcaster->SetIngestServer(ingestServer);
	
	if (TTV_SUCCEEDED(ec)) {
		
		_ingestServer = ingestServer;
	}
	else NSLog(@"TWSTREAMER: Failed to set ingest server");
}

@end
